import 'dart:convert';
import 'package:admin_desktop/domain/di/dependency_manager.dart';
import 'package:flutter/material.dart';
import 'package:admin_desktop/domain/handlers/handlers.dart';
import 'package:admin_desktop/domain/models/response/stories_response.dart';
import 'package:admin_desktop/infrastructure/services/utils.dart';
import '../stories_repository.dart';

class StoriesRepositoryImpl extends StoriesRepository {
  @override
  Future<ApiResult<StoriesResponse>> getStories({int? page}) async {
    final data = {
      'page': page,
      'perPage': 12,
      'lang': LocalStorage.getLanguage()?.locale ?? 'en',
      'shop_id': LocalStorage.getUser()?.shop?.id,
    };
    try {
      final client = dioHttp.client(requireAuth: true);
      final response = await client.get(
        "/api/v1/dashboard/seller/stories",
        queryParameters: data,
      );
      return ApiResult.success(data: StoriesResponse.fromJson(response.data));
    } catch (e, s) {
      debugPrint('==> get stories failure: $e,$s');
      return ApiResult.failure(error: AppHelpers.errorHandler(e));
    }
  }

  @override
  Future<ApiResult<void>> deleteStories(int id) async {
    final data = {
      'ids': [id],
    };
    debugPrint('====> delete brand request ${jsonEncode(data)}');
    try {
      final client = dioHttp.client(requireAuth: true);
      await client.delete(
        '/api/v1/dashboard/seller/stories/delete',
        data: data,
      );
      return const ApiResult.success(data: null);
    } catch (e) {
      debugPrint('==> delete brand failure: $e');
      return ApiResult.failure(error: AppHelpers.errorHandler(e));
    }
  }

  @override
  Future<ApiResult<void>> createStories({
    required List<String> images,
    int? serviceId,
    int? productId,
  }) async {
    final data = {
      'file_urls': images,
      if (serviceId != null || productId != null)
        'model_id': serviceId ?? productId,
      'model_type': serviceId != null
          ? 'service'
          : productId != null
          ? 'product'
          : 'shop',
    };
    debugPrint('====> add stories request ${jsonEncode(data)}');
    try {
      final client = dioHttp.client(requireAuth: true);
      await client.post('/api/v1/dashboard/seller/stories', data: data);
      return const ApiResult.success(data: null);
    } catch (e) {
      debugPrint('==> create stories failure: $e');
      return ApiResult.failure(error: AppHelpers.errorHandler(e));
    }
  }

  @override
  Future<ApiResult<void>> updateStories({
    required List<String> images,
    int? id,
    String? modelType,
    required int storyId,
  }) async {
    final data = {
      'file_urls': images,
      if (id != null) 'model_id': id,
      if (modelType != null) 'model_type': modelType,
    };
    debugPrint('====> update stories request ${jsonEncode(data)}');
    try {
      final client = dioHttp.client(requireAuth: true);
      await client.put('/api/v1/dashboard/seller/stories/$storyId', data: data);
      return const ApiResult.success(data: null);
    } catch (e) {
      debugPrint('==> update stories failure: $e');
      return ApiResult.failure(error: AppHelpers.errorHandler(e));
    }
  }
}
